<?php

/**
 * ARK Manager File
 *  
 * Contains the ARK Manager code 
 *
 * @author Robbie Hott
 * @license http://opensource.org/licenses/BSD-3-Clause BSD 3-Clause
 * @copyright 2015 the Rector and Visitors of the University of Virginia, and
 *            the Regents of the University of California
 */
namespace ark;

/**
 * ARK Manager
 *
 * This class manages ARK ids, including the assigning of real arks as well as the minting of temporary
 * arks for testing purposes.
 *
 * @author Robbie Hott
 */
class ArkManager {

    /**
     * @var \ark\database\DatabaseConnector The helper class object for database connections
     */
    private $db = null;

    /**
     * Constructor
     *
     * Sets up the ArkManager
     */
    public function __construct() {
        $this->db = new \ark\database\DatabaseConnector();
    }

    /**
     * Mint Temporary Ark
     *
     * Mint arks used for testing purposes.  These are not "valid" arks but are randomly generated based on
     * a seed of the current timestamp.  They should not be used for production purposes and are not guaranteed
     * to resolve.
     *
     * @param integer $num optional The number of arks to generate. If more than one, this method will return an array
     * @return string|string[]|null A temporary ark or a list of temporary arks
     */
    public function mintTemporaryArk($num=1) {
        if (!is_numeric($num) || $num <= 0) 
            return null;

        // Set up minting factory
        $prefix = Config::$ARK_URL . "/ark:/" . Config::$TEMPORARY_INSTITUTION_CODE . "/Z";

        if ($num == 1)
            return $prefix . $this->generateRandomString();

        $arks = array();
        for ($i = 0; $i < $num; $i++) {
            // Store a temporary ark
            $ark = $prefix . $this->generateRandomString();
            array_push($arks, $ark);
        }
        return $arks;
    }

    /**
     * Mint Real Ark
     *
     * Mint a real, unique, permanent, resolvable ark.  Once an ark is minted using this method, it will never be reassigned. This
     * process is irreversible, so use with caution.  Because of this, only one may be requested at a time.
     *
     * @return string|boolean A new permanent arkID or false if an ark could not be minted
     */
    public function mintArk() {
        try {
            // get the first ark in the list while deleting it (returning *) so that it is atomic
            $resource = $this->db->query("delete from unassigned_arks 
                                    where id = (select min(id) from unassigned_arks) 
                                    returning *;", array());
            $result = $this->db->fetchRow($resource);

            // If the result exists, is not false, and has an ark, then return the ark
            if ($result !== false && is_array($result) && isset($result['ark']))
                return $result['ark'];

        } catch (\Exception $e) {
            // Something went wrong, so no ark can be returned
            return false;
        }
        
        return false;
    }

    /**
     * Generate a random string
     *
     * Generate a random ASCII string of a given length.  The default length is 7 for temporary ARK URL generation.
     *
     * @param integer $length optional The length of string to generate
     * @param string A random ASCII string
     */
    private function generateRandomString($length=7) {
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $cLength = strlen($characters);
        $string = '';
        for ($i = 0; $i < $length; $i++) {
            $string .= $characters[random_int(0, $cLength - 1)];
        }
        return $string;
    }

}
